/*
  Pre-filter housing spanner wrench - NOTCH (凹槽) type (CLEAN boolean)
  ✅ 6个凹槽均匀分布
  ✅ 凹槽切除块向内孔多切一点，避免“圆弧盖住凹槽”的脏边界
  - thickness 可调
  - handle_len 可调
*/

$fn = 720;              // 圆更顺滑（可再加大）

// -------------------- Main parameters --------------------
thickness   = 10;        // 总厚度/挤出厚度 (mm)  <-- 可调
handle_len  = 150;      // 力臂长度/手柄长度 (mm) <-- 可调

// Ring geometry (3D打印建议留点间隙)
inner_d_nom = 58;       // 标称内径 (mm)
fit_clear   = 0.6;      // ✅ 打印装配间隙(0.3~1.0mm常用)，越大越松
inner_d     = inner_d_nom + fit_clear;

ring_width  = 12;       // 环形外圈宽度 (mm)
outer_d     = inner_d + 2*ring_width;

// ✅ Notches (recesses) that match bumps on the housing
notch_count = 6;        // 6个凹槽（均匀）
notch_depth = 3.0;      // 凹槽向外吃掉的深度 (mm)
notch_width = 5;      // 凹槽沿圆周方向宽度 (mm)
notch_angle0= 0;        // 对不准凸起就改这个角度（deg）

notch_in    = 1.2;      // ✅ 关键：向内孔方向“多切”1~2mm，保证布尔干净
eps         = 0.05;     // ✅ 防止几何重合的小量

// Handle geometry
handle_w0   = 26;
handle_w1   = 18;
handle_round= 10;

// Reinforcement / comfort
neck_extra  = 12;
hole_d      = 8;        // <=0 不打孔


// -------------------- 2D primitives --------------------
module ring_2d() {
  difference() {
    circle(d=outer_d);
    circle(d=inner_d);
  }
}

// 单个“凹槽/缺口”的切除块（difference里减掉）
// ✅ 注意：切除块尺寸 = notch_depth + notch_in，并且位置让它“侵入”内孔 notch_in
module notch_cut_2d() {
  r = inner_d/2;
  cut_len = notch_depth + notch_in + 2*eps;

  // 让切除块覆盖区间：[r - notch_in - eps, r + notch_depth + eps]
  // center 放在 r + (notch_depth - notch_in)/2
  cx = r + (notch_depth - notch_in)/2;

  translate([cx, 0])
    square([cut_len, notch_width + 2*eps], center=true);
}

module handle_2d() {
  y0 = -outer_d/2;
  y1 = y0 - handle_len;

  polygon(points=[
    [-(handle_w0/2), y0],
    [ (handle_w0/2), y0],
    [ (handle_w1/2), y1 + handle_round],
    [ (handle_w1/2), y1],
    [-(handle_w1/2), y1],
    [-(handle_w1/2), y1 + handle_round]
  ]);
}

module handle_end_round_2d() {
  y0 = -outer_d/2;
  y1 = y0 - handle_len;
  translate([0, y1 + handle_round])
    circle(r=handle_round);
}

module drill_hole_2d() {
  if (hole_d > 0) {
    y0 = -outer_d/2;
    y1 = y0 - handle_len;
    translate([0, y1 + handle_round])
      circle(d=hole_d);
  }
}

// -------------------- Main 2D shape --------------------
module wrench_2d() {
  difference() {
    union() {
      ring_2d();
      handle_2d();
      handle_end_round_2d();

      // 脖子加强
      translate([0, -outer_d/2 - 6])
        offset(r=2)
          square([handle_w0 + neck_extra, 14], center=true);
    }

    // ✅ 6个凹槽均匀分布
    for (i = [0:notch_count-1]) {
      rotate(notch_angle0 + i*360/notch_count)
        notch_cut_2d();
    }
  }
}

// -------------------- Final 3D --------------------
difference() {
  // render() 让布尔结果更“干净”（预览也更一致）
  render(convexity=10)
    linear_extrude(height=thickness)
      wrench_2d();

  // 手柄挂孔（贯穿）
  if (hole_d > 0)
    translate([0,0,-1])
      linear_extrude(height=thickness+2)
        drill_hole_2d();
}
